// Example0601 - フォトリフレクタとロボットの動作
// LEDの変数
const int LED_R_PIN = 4; // LED赤のピン
const int LED_G_PIN = 3; // LED緑のピン
const int LED_Y_PIN = 13; // LED黄のピン
unsigned long dist_time = 0; // 距離センサ用のタイマー

// モーターの変数
const int MOTOR_R_CWCCW = 7;
const int MOTOR_L_CWCCW = 8;
const int MOTOR_R_PWM = 9;
const int MOTOR_L_PWM = 10;
const int motor_sp = 100; // モーター左右のスピード: 0 - 255

// スイッチの変数
const int SW_PIN = 12;

// フォトリフレクタの変数
#define PR_NUM   3 // フォトリフレクタの数
int pr_pins[PR_NUM] = {A2, A1, A0}; // フォトリフレクタのピン(左，中，右)
int pr[PR_NUM]; // フォトリフレクタの値
int pr_bin[PR_NUM]; // フォトリフレクタの値（2値化後)
int pr_min[PR_NUM], pr_max[PR_NUM]; // フォトリフレクタの最小値，最大値
unsigned long pr_time = 0; // フォトリフレクタのタイマー

void setup() {
  // シリアルポートの設定
  Serial.begin(38400); // bps for bluetooth communication
  Serial.flush(); // flush serial buffer

  // LEDのポート設定
  pinMode(LED_R_PIN, OUTPUT);
  pinMode(LED_G_PIN, OUTPUT);
  pinMode(LED_Y_PIN, OUTPUT);

  // モータのポート設定
  pinMode(MOTOR_L_CWCCW, OUTPUT);
  pinMode(MOTOR_R_CWCCW, OUTPUT);

  // スイッチが押されるまで待機
  while (digitalRead(SW_PIN) == HIGH);
}

void loop() {
  // フォトリフレクタのセンサの値取得
  if (millis() - pr_time > 50) { // フォトリフレクタ更新(50ミリ秒毎)
    pr_sensor();
  }

  // ロボットの動作
  // 条件を変えてみる
  // すべてがライン上にきたら pr_l && pr_c && pr_r
  // どれか一つでもライン上にきたら pr_l || pr_c ||  pr_r
  if (pr_bin[0] == 1 && pr_bin[1] == 1 && pr_bin[2] == 1) {
    bwd(2.0); // 少し後退
    rotL(180.0); // 左旋回
  } else {
    fwd(1.0); // 直進
  }
}

//===========================================
// LEDの関数 赤(左)，緑(右)，黄(電源横)
//===========================================
void led_on(bool r, bool g, bool y) {
  digitalWrite(LED_R_PIN, r);
  digitalWrite(LED_G_PIN, g);
  digitalWrite(LED_Y_PIN, y);
}

//===========================================
// フォトリフレクタの関数
//===========================================
int pr_sensor() {
  for (int i = 0; i < PR_NUM; i++) {
    pr[i] = analogRead(pr_pins[i]);
    if (pr[i] > 700) {  // 閾値
      pr_bin[i] = 1; // ライン上
    } else {
      pr_bin[i] = 0; // ライン外
    }
    Serial.print(pr_bin[i]);
    Serial.print(",");
  }
  Serial.println();
}

//===========================================
// 距離センサの関数 定数は各自のセンサで校正
//===========================================
int dist_sensor() {
  int sensorValue = analogRead(A3); // read value from distance sensor
  double dist = 322549.9 * pow(sensorValue, -1.25356); // calcurate distance [mm]
  return dist;
}

//===========================================
// モーターの関数
//===========================================
void motor(int left, int right, int left_c, int right_c) {
  if (left_c == LOW) {
    digitalWrite(MOTOR_L_CWCCW, LOW);
  } else {
    digitalWrite(MOTOR_L_CWCCW, HIGH);
  }
  if (right_c == LOW) {
    digitalWrite(MOTOR_R_CWCCW, LOW);
  } else {
    digitalWrite(MOTOR_R_CWCCW, HIGH);
  }
  analogWrite(MOTOR_L_PWM, left);
  analogWrite(MOTOR_R_PWM, right);
}

void fwd(float cm) {
  int tau;
  led_on(1,1,0);
  motor(motor_sp, motor_sp, LOW, LOW);
  tau = 2500.0 * cm / 20.0;
  delay(tau);
}

void bwd(float cm) {
  int tau;
  led_on(0,0,1);
  motor(motor_sp, motor_sp, HIGH, HIGH);
  tau = 2500.0 * cm / 20.0;
  delay(tau);
}

void stp(int tau) {
  led_on(0,0,0);
  motor(0, 0, LOW, LOW);
  delay(tau);
}

void rotL(float deg) {
  int tau;
  led_on(1,0,0);
  motor(motor_sp, motor_sp, HIGH, LOW);
  tau = 3780.0 * deg / 360.0;
  delay(tau);
}

void rotR(float deg) {
  int tau;
  led_on(0,1,0);
  motor(motor_sp, motor_sp, LOW, HIGH);
  tau = 3820.0 * deg / 360.0;
  delay(tau);
}

