// Example0606 ロボットをライン上に置いてスイッチを押し自動校正
// LEDの変数
const int LED_R_PIN = 4; // LED赤のピン
const int LED_G_PIN = 3; // LED緑のピン
const int LED_Y_PIN = 13; // LED黄のピン

// モーターの変数
const int MOTOR_R_CWCCW = 7;
const int MOTOR_L_CWCCW = 8;
const int MOTOR_R_PWM   = 9;
const int MOTOR_L_PWM   = 10;
#define FWD  LOW
#define BWD  HIGH

// フォトリフレクタの変数
#define PR_NUM  3 // フォトリフレクタの数
int pr_pins[PR_NUM] = {A2, A1, A0}; // フォトリフレクタのピン(左，中，右)
int pr[PR_NUM]; // フォトリフレクタの値
int pr_nrm[PR_NUM]; // 正規化後のフォトリフレクタの値0-100
int pr_bin[PR_NUM]; // フォトリフレクタの値(2値化後)
int pr_min[PR_NUM], pr_max[PR_NUM]; // フォトリフレクタの最小値，最大値
//int  pr_min[PR_NUM] = {457, 414, 614}; // 自動校正した最小値を貼り付け
//int  pr_max[PR_NUM] = {950, 841, 996}; // 自動校正した最大値を貼り付け
unsigned long pr_time = 0; // フォトリフレクタ用のタイマー

// スイッチの変数
const int SW_PIN = 12;

void setup() {
  // シリアルポートの設定
  Serial.begin(38400); // bps for bluetooth communication
  Serial.flush(); // flush serial buffer

  // LEDのポート設定
  pinMode(LED_R_PIN, OUTPUT);
  pinMode(LED_G_PIN, OUTPUT);
  pinMode(LED_Y_PIN, OUTPUT);

  // モータのポート設定
  pinMode(MOTOR_L_CWCCW, OUTPUT);
  pinMode(MOTOR_R_CWCCW, OUTPUT);

  // スイッチが押されるまで待機
  while (digitalRead(SW_PIN) == HIGH);

  // 自動校正開始
  pr_calib(); // フォトリフレクタの自動校正(校正完了後はコメントアウト)
}

void loop() {
  if (millis() - pr_time > 50) { // フォトリフレクタのセンサ値：50ミリ秒毎取得
    for (int i = 0; i < PR_NUM; i++) {
      pr[i] = analogRead(pr_pins[i]); // センサの値取得A2,A1,A0
      if (pr[i] < pr_min[i]) pr_min[i] = pr[i]; // 最小値を更新
      if (pr[i] > pr_max[i]) pr_max[i] = pr[i]; // 最大値を更新
      pr_nrm[i] = 100.0 * (pr[i] - pr_min[i]) / (pr_max[i] - pr_min[i]); // 0%～100%に正規化
      if (pr_nrm[i] > 50) pr_bin[i] = 1; // 50%より大きい1, 以下は0として2値化
      else pr_bin[i] = 0;
    }
  }
  // それぞれグラフで確認：1:pr, 2:pr_nrm, 3:pr_bin
  //pr_print(1);
  //pr_print(2);
  //pr_print(3);
}

void pr_print(int data) {
  if (data == 1) { // pr センサの値
    Serial.print(pr[0]);
    Serial.print(",");
    Serial.print(pr[1]);
    Serial.print(",");
    Serial.println(pr[2]);
  } else if (data == 2) { // pr_nrm 正規化後の値
    Serial.print(pr_nrm[0]);
    Serial.print(",");
    Serial.print(pr_nrm[1]);
    Serial.print(",");
    Serial.println(pr_nrm[2]);
  } else { // pr_bin 2値化後の値
    Serial.print(pr_bin[0]);
    Serial.print(",");
    Serial.print(pr_bin[1]);
    Serial.print(",");
    Serial.println(pr_bin[2]);
  }
}

void pr_minmax() {
  for (int i = 0; i < PR_NUM; i++) {
    pr[i] = analogRead(pr_pins[i]); // センサの値取得
    if (pr[i] < pr_min[i]) pr_min[i] = pr[i]; // 最小値の更新
    if (pr[i] > pr_max[i]) pr_max[i] = pr[i]; // 最大値の更新
  }
}

void pr_calib() {
  // 最小，最大値の初期化
  for (int i = 0; i < PR_NUM; i++) {
    pr_min[i] = 1023; // 最小値には，最大値をセット
    pr_max[i] = 0; // 最大値には，最小値をセット
  }

  // ロボットをライン上で左右に動かし，各センサの最小値，最大値を探す
  for (int i = 0; i < 80; i++) {
    if ( (i > 10 && i <= 30) || (i > 50 && i <= 70) ) {
      motor(100, 100, FWD, BWD); // 左旋回
    } else {
      motor(100, 100, BWD, FWD); // 右旋回
    }
    pr_minmax(); // 最小値，最大値の更新
    delay(40);
  }
  motor(0, 0, FWD, FWD); // 停止

  pr_print_minmax(); // 最小値，最大値の配列を表示
}

// フォトリフレクタの最小値，最大値の表示
// 校正後，プログラムに貼り付けて使用
void pr_print_minmax() {
  Serial.print("int pr_min[PR_NUM] = {");
  Serial.print(pr_min[0]);
  Serial.print(",");
  Serial.print(pr_min[1]);
  Serial.print(",");
  Serial.print(pr_min[2]);
  Serial.println("};");

  Serial.print("int pr_max[PR_NUM] = {");
  Serial.print(pr_max[0]);
  Serial.print(",");
  Serial.print(pr_max[1]);
  Serial.print(",");
  Serial.print(pr_max[2]);
  Serial.println("};");
}

void motor(int left, int right, int left_c, int right_c) {
  if (left_c == FWD) {
    digitalWrite(MOTOR_L_CWCCW, FWD);
  } else {
    digitalWrite(MOTOR_L_CWCCW, BWD);
  }
  if (right_c == FWD) {
    digitalWrite(MOTOR_R_CWCCW, FWD);
  } else {
    digitalWrite(MOTOR_R_CWCCW, BWD);
  }
  analogWrite(MOTOR_L_PWM, left); // PWM出力
  analogWrite(MOTOR_R_PWM, right); // PWM出力
}

