// Example0705 ライントレース(PD制御) センサ位置
const int LED_R_PIN = 4; // LED赤のピン
const int LED_G_PIN = 3; // LED緑のピン
const int LED_Y_PIN = 13; // LED黄のピン

// モーターの変数
const int MOTOR_R_CWCCW = 7;
const int MOTOR_L_CWCCW = 8;
const int MOTOR_R_PWM   = 9;
const int MOTOR_L_PWM   = 10;
#define FWD  LOW
#define BKW  HIGH
const int pos_t = 100; // 目標値：100
const int sp = 100; // モーターの基本スピード: 0 - 255

// PD制御用
float KP = 0.2; // 0.1 0-200
float KD = 0.1; // 0-200
int last_error = 0; // 目標値との誤差
int last_pos = 0;   // 前回の位置

// フォトリフレクタの変数
#define PR_NUM  3
int pr_pins[PR_NUM] = {A2, A1, A0};
int pr[PR_NUM];
int pr_nrm[PR_NUM]; // 正規化後のフォトリフレクタの値0-100
int pr_bin[PR_NUM]; // フォトリフレクタの値(2値化後)
//int pr_min[PR_NUM], pr_max[PR_NUM]; // フォトリフレクタの最小値，最大値
int  pr_min[PR_NUM] = {457, 414, 614}; // 自動校正した最小値を貼り付け(Example0606)
int  pr_max[PR_NUM] = {950, 841, 996}; // 自動校正した最大値を貼り付け(Example0606)
unsigned long pr_time = 0; // フォトリフレクタ用のタイマー

// スイッチの変数
const int SW_PIN = 12;

void setup() {
  // シリアルポートの設定
  Serial.begin(38400); // bps for bluetooth communication
  Serial.flush(); // flush serial buffer

  // LEDのポート設定
  pinMode(LED_R_PIN, OUTPUT);
  pinMode(LED_G_PIN, OUTPUT);
  pinMode(LED_Y_PIN, OUTPUT);

  // モータのポート設定
  pinMode(MOTOR_L_CWCCW, OUTPUT);
  pinMode(MOTOR_R_CWCCW, OUTPUT);

  while (digitalRead(SW_PIN) == HIGH); // スイッチが押されるまで待機

  // フォトリフレクタの校正
//  pr_calib();
}

void loop() {
  if (millis() - pr_time > 50) {
    pr_sensor(); // フォトリフレクタのセンサ値更新
    int pos = pr_pos(); // フォトリフレクタの位置(0-200)
    pd(pos); // モーター制御
    pr_time = millis();
  }
}

//===========================================
// モーター制御の関数
//===========================================
void pd(int pos) {
  int error = pos_t - pos; // 目標値(100)との計測位置の差
  int sp_diff = KP * error + KD * (error - last_error); // PD制御
  last_error = error; // 前の誤差

  int sp_l = sp - sp_diff; // 左モーターの速度
  int sp_r = sp + sp_diff; // 右モーターの速度

  // 左，右モーターの最小値，最大値を超えないか確認
  if (sp_l < 0)   sp_l = 0;
  if (sp_r < 0)   sp_r = 0;
  if (sp_l > 255) sp_l = 255;
  if (sp_r > 255) sp_r = 255;

  motor(sp_l, sp_r, FWD, FWD);
}


//===========================================
// フォトリフレクタの関数
//===========================================
// フォトリフレクタの位置を0～200で返す
// センター：100
// 100,110,010,011,001
//  0, 50, 100,150,200
int pr_pos() {
  int pos, sum, avg;

  avg = 0;
  sum = 0;
  for (int i = 0; i < PR_NUM; i++) {
    avg += pr_bin[i] * (i * 100);
    sum += pr_bin[i];
  }

  if (sum == 0) { // ライン外
    return last_pos;
  }
  pos = avg / sum;
  last_pos = pos; // 直前の位置

  return pos;
}

// フォトリフレクタのセンサ値 pr[]: 0-1023
// フォトリフレクタの正規化値 pr_nrm[]: 0-100
// フォトリフレクタの2値化 pr_bin[]: 0, 1
void pr_sensor() {
  for (int i = 0; i < PR_NUM; i++) {
    pr[i] = analogRead(pr_pins[i]);
    if (pr[i] < pr_min[i]) pr_min[i] = pr[i];
    if (pr[i] > pr_max[i]) pr_max[i] = pr[i];
    pr_nrm[i] = 100.0 * (pr[i] - pr_min[i]) / (pr_max[i] - pr_min[i]);
    if (pr_nrm[i] > 50) pr_bin[i] = 1;
    else pr_bin[i] = 0;
  }
}

void pr_minmax() {
  for (int i = 0; i < PR_NUM; i++) {
    pr[i] = analogRead(pr_pins[i]);
    if (pr[i] < pr_min[i]) pr_min[i] = pr[i];
    if (pr[i] > pr_max[i]) pr_max[i] = pr[i];
  }
}

void pr_calib() {
  for (int i = 0; i < PR_NUM; i++) {
    pr_min[i] = 1023;
    pr_max[i] = 0;
  }

  for (int i = 0; i < 80; i++) {
    if ( (i > 10 && i <= 30) || (i > 50 && i <= 70) ) {
      motor(150, 150, FWD, BKW);
    } else {
      motor(150, 150, BKW, FWD);
    }
    pr_minmax();
    delay(40);
  }
  motor(0, 0, FWD, FWD);
}

void motor(int left, int right, int left_c, int right_c) {
  if (left_c == FWD) {
    digitalWrite(MOTOR_L_CWCCW, FWD);
  } else {
    digitalWrite(MOTOR_L_CWCCW, BKW);
  }
  if (right_c == FWD) {
    digitalWrite(MOTOR_R_CWCCW, FWD);
  } else {
    digitalWrite(MOTOR_R_CWCCW, BKW);
  }
  analogWrite(MOTOR_L_PWM, left); // PWM出力
  analogWrite(MOTOR_R_PWM, right); // PWM出力
}

