// Ex0905: ライントレース(PD制御) + EEPROM(最小・最大)
// ・迷路パターン8種類のチェックポイント確認
// ・迷路の通過経路の記録 maze_pattern[]
// ・迷路の最短経路を算出 find_path();
// ・ゴールアクション goal_action();
// ・迷路の探索(最短) maze_solve();
// ＊迷路を左手法で走行した後，最短経路で走らせてみる
#include <FastLED.h>
#include <EEPROM.h>

// LED 16bit
const int DATA_PIN = 19;
const int NUM_LEDS = 16;
CRGB leds[NUM_LEDS];

// PD制御用
// LIMIT:500-TARGET:300 (+-200の範囲)
float KP = 1.8; // 比例制御
float KD = 0.2; // 微分制御
#define LIMIT 500

// モーターの変数
const int MOTOR_L_CWCCW = 16; // APH direction
const int MOTOR_R_CWCCW = 17; // BPH direction
const int MOTOR_L_PWM = 26; // AEN pwm
const int MOTOR_R_PWM = 4; // BEN pwm

// PWM
const int CH_0 = 0;
const int CH_1 = 1;
const int LEDC_TIMER_BIT = 10;   // PWMの範囲(10bit: 0-1023)
const int LEDC_BASE_FREQ = 10000; // 周波数(Hz) 100khz
const int VALUE_MAX = 1023;      // PWMの最大値
const int sp  = 300; // モーターの基本スピード: 0 - 1023

#define NUM_PR  5 // センサの数
const int pr_pins[NUM_PR] = {A3, A6, A7, A4, A5}; // ピン定義
int pr[NUM_PR] = {0};
int pr_bin[NUM_PR] = {0}; // ラインの有無
int pr_min[NUM_PR] = {4095, 4095, 4095, 4095, 4095}; // 最小値
int pr_max[NUM_PR] = {0}; // 最大値
float pr_nrm[NUM_PR] = {0}; // 正規化0.0-1.0
unsigned long pr_time = 0; // フォトインタラプタ用タイマー計測値
const int pos_t = 300; // 目標値

// スイッチの変数
const int SW_PIN = 25;
int sw_calib = 0;

void setup() {
  // シリアルポートの設定
  Serial.begin(115200); // bps for bluetooth communication
  delay(100);

  // EEPROM
  EEPROM.begin(100); // EEPROMに100byteの変数用の領域確保

  // モータのポート設定
  pinMode(MOTOR_L_CWCCW, OUTPUT);
  pinMode(MOTOR_R_CWCCW, OUTPUT);

  // PWM
  ledcSetup(CH_0, LEDC_BASE_FREQ, LEDC_TIMER_BIT);
  ledcSetup(CH_1, LEDC_BASE_FREQ, LEDC_TIMER_BIT);
  ledcAttachPin(MOTOR_L_PWM, CH_0);
  ledcAttachPin(MOTOR_R_PWM, CH_1);
  ledcWrite(CH_0, 0);
  ledcWrite(CH_1, 0);

  // led
  FastLED.addLeds<WS2812B, DATA_PIN, GRB>(leds, NUM_LEDS);
  FastLED.setBrightness(32); // 最大の明るさ 0-255

  // スイッチ
  pinMode(SW_PIN, INPUT_PULLUP);

  // 自動校正の有無の確認
  auto_calib();
  //  pr_print_minmax();

  // スイッチを押すまで待機
  while (digitalRead(SW_PIN) == HIGH);
}

#define D_TIME (100) // 少し前に進む時間，書き換える
#define L_TIME (300) // 左90度旋回時間，書き換える
#define R_TIME (300) // 右90度旋回時間，書き換える
#define T_TIME (530) // 左180度旋回時間，書き換える
#define I_TIME (5)   // 交差点の判定時間，書き換える
#define A_TIME (120) // ラインを避ける時間，書き換える

void loop() {
  while (maze_pattern()); // 迷路のゴールに到達するまで
  goal_action(); // ゴールアクション
  check_path(); // 迷路の経路確認

  while (maze_solve()); // 迷路の探索(最短)
  goal_action(); // ゴールアクション
  check_path(); // 迷路の経路確認
}

//===========================================
// 迷路探索の関数
//===========================================
char maze_path[50] = {0}; // 迷路の経路
int path_len = 0; // 迷路の経路長

int maze_pattern() {
  // ライントレース
  follow_line();

  // 交差点の判定
  if (pr_bin[0] == 1 || pr_bin[4] == 1) {
    delay(I_TIME);
    get_pr();
  }

  // ⑧行き止まり
  if (pr_bin[0] == 0 && pr_bin[1] == 0 && // ラインなし
      pr_bin[2] == 0 && pr_bin[3] == 0 &&
      pr_bin[4] == 0) {
    maze_path[path_len] = 'T';
    path_len++;
    go_D(D_TIME);
    turn_L(T_TIME);
  }
  // ③T字，④四角，⑦ゴール
  else if (pr_bin[0] == 1 && pr_bin[4] == 1) { // 左端，右端のセンサ反応
    go_D(D_TIME);
    get_pr();
    if (pr_bin[0] == 1 && pr_bin[1] == 1 && // センサ全反応
        pr_bin[2] == 1 && pr_bin[3] == 1 &&
        pr_bin[4] == 1) {
      return 0;  // ゴール
    } else {
      maze_path[path_len] = 'L';
      path_len++;
      turn_L(L_TIME);
    }
  }
  // ①左折，⑤ト字路
  else if (pr_bin[0] == 1) { // 左折
    maze_path[path_len] = 'L';
    path_len++;
    go_D(D_TIME);
    turn_L(L_TIME);
  }
  // ②右折，⑥ト字路
  else if (pr_bin[4] == 1) { // 直進，右折
    go_D(D_TIME);
    get_pr();
    if (pr_bin[0] == 0 && // ラインの真ん中あたり
        (pr_bin[1] == 1 || pr_bin[2] == 1 || pr_bin[3] == 1 ) &&
        pr_bin[4] == 0) {
      maze_path[path_len] = 'S';
      path_len++;
    }
    if ( pr_bin[0] == 0 && pr_bin[1] == 0 && // ラインなし
         pr_bin[2] == 0 && pr_bin[3] == 0 &&
         pr_bin[4] == 0) {
      maze_path[path_len] = 'R';
      path_len++;
      turn_R(R_TIME);
    }
  }

  find_path(); // 最短経路の算出

  return 1;
}

// 迷路の探索(最短)
int maze_solve() {
  static int opt_path;

  // ライントレース
  follow_line();

  // 交差点の判定
  if (pr_bin[0] == 1 || pr_bin[4] == 1) {
    delay(I_TIME);
    get_pr();

    // 記録した最短経路に合わせて動かす
    switch (maze_path[opt_path]) {
      case 'L':
        go_D(D_TIME);
        turn_L(L_TIME);
        break;
      case 'R':
        go_D(D_TIME);
        turn_R(R_TIME);
        break;
      case 'S':
        go_D(D_TIME);
        break;
      default:
        break;
    }
    opt_path++;
  }

  // 全パスを走行したら
  if (opt_path >= path_len) {
    // ゴールまで走る
    while (1) {
      follow_line();
      if (pr_bin[0] == 1 && pr_bin[1] == 1 &&
          pr_bin[2] == 1 && pr_bin[3] == 1 &&
          pr_bin[4] == 1) {
        go_D(D_TIME);
        return 0; // ゴール：終了
      }
      if (pr_bin[0] == 0 && pr_bin[1] == 0 &&
          pr_bin[2] == 0 && pr_bin[3] == 0 &&
          pr_bin[4] == 0) {
        opt_path = 0;
        check_path();
        return 1; // 行き止まり：やり直し
      }
      delay(1);
    }
  } else {
    return 1;
  }
}

// ゴールアクション
void goal_action() {
  motor_stop(1000);
  // ロボットを回転，LEDを点滅，音を鳴らすなど考える
}

// 無駄経路の削除
// maze_path[]={L,T,L}, path_len = 3
// LTL => 270+180+270=720 => 720/360=2 => 0('S')
// [S]TL 上書き, path_len-=2
// [S], path_len=1
void find_path() {
  int total_angle = 0;

  // パスが3未満のとき，パスの2番目に’T'が含まれないときは処理しない
  if (path_len < 3 || maze_path[path_len - 2] != 'T') {
    return;
  }

  // 経路の向きの角度を計算：直近の3パス分
  for (int i = 1; i <= 3; i++) {
    switch (maze_path[path_len - i]) {
      case 'R':
        total_angle += 90;
        break;
      case 'L':
        total_angle += 270;
        break;
      case 'T':
        total_angle += 180;
        break;
      default:
        break;
    }
  }

  // 3パス分の経路の角度を，360度で割った余り
  total_angle = total_angle % 360; // 余りの計算
  // 経路を上書きする
  switch (total_angle) {
    case 0:
      maze_path[path_len - 3] = 'S';
      break;
    case 90:
      maze_path[path_len - 3] = 'R';
      break;
    case 180:
      maze_path[path_len - 3] = 'T';
      break;
    case 270:
      maze_path[path_len - 3] = 'L';
      break;
    default:
      break;
  }
  path_len -= 2; // 上書きした経路の次から続きを始める
}

// 経路の確認
void check_path() {
  motor_stop(1000);
  while (digitalRead(SW_PIN)); // スイッチが押されるまで待つ
  Serial.println(path_len);
  for (int i = 0; i < path_len; i++) { // 迷路の経路を表示
    Serial.println(maze_path[i]);
  }
  delay(1000);
  while (digitalRead(SW_PIN)); // スイッチが押されるまで待つ
}

//===========================================
// ロボットの動作関数
//===========================================
void follow_line() {
  get_pr(); // フォトリフレクタのセンサ値更新
  int pos = pr_pos(); // フォトリフレクタの位置(100-500)
  pd(pos); // モーター制御
}

void go_D(int d) {
  motor(sp, sp, HIGH, HIGH); // forward
  delay(d);
  motor_stop(300);
}

void turn_L(int d) {
  motor(sp, sp, LOW, HIGH); // left
  delay(A_TIME);
  for (int i = 0; i < d; i++) {
    get_pr();
    if (pr_bin[2] == 1) break;
    delay(1);
  }
}

void turn_R(int d) {
  motor(sp, sp, HIGH, LOW); // right
  delay(A_TIME);
  for (int i = 0; i < d; i++) {
    get_pr();
    if (pr_bin[2] == 1) break;
    delay(1);
  }
}

//===========================================
// 自動校正の関数
//===========================================
void auto_calib() {
  int sw_calib = 0; // 自動校正の有無

  // 起動時に校正の有無を確認
  if (digitalRead(SW_PIN) == LOW) { // スイッチが押されていたら
    sw_calib = 1; // 校正をON
    for (int i = 0; i < 3; i++) { // 3回白色点滅
      leds[0] = CRGB(255, 255, 255);
      FastLED.show();
      delay(200);
      leds[0] = CRGB(0, 0, 0);
      FastLED.show();
      delay(200);
    }
  }

  if (sw_calib == 1) {  // 自動校正開始
    sw_calib = 0;
    pr_calib(); // フォトリフレクタの自動校正
    eeprom_write(); // EEPROMに校正値を書き込み
  } else {
    eeprom_read(); // EEPROMから校正値を読み込み
  }
}

// EEPROMに最小・最大値の書き込み
void eeprom_write() {
  int n = 0;
  for (int i = 0; i < NUM_PR; i++) {
    EEPROM.put(n, pr_min[i]);
    EEPROM.put(n + 4 * NUM_PR, pr_max[i]);
    n += 4;
  }
  EEPROM.commit(); // EEPROMに書き込み
}

// EEPROMより最小・最大値の読み込み
void eeprom_read() {
  int n = 0;
  for (int i = 0; i < NUM_PR; i++) {
    EEPROM.get(n, pr_min[i]);
    EEPROM.get(n + 4 * NUM_PR, pr_max[i]);
    n += 4;
  }
  EEPROM.end();
}

//===========================================
// フォトリフレクタの関数
//===========================================
// フォトリフレクタの位置を100～500で返す
// ライン外:0
// センター：300
// LEFT-----------CENTER-----------RIGHT
// 100,150,200,250,<300>,350,400,450,500
// OUT of LINE: 0
int pr_pos() {
  int pos;
  static int last_pos;

  int sum = 0, n = 0;
  for (int i = 0; i < NUM_PR; i++) {
    sum += pr_bin[i] * ((i + 1) * 100);
    if (pr_bin[i] == 1) n++;
  }

  if (n > 0) { // ラインの上
    pos = sum / n;
  } else {
    pos = 0;
  }

  return pos;
}

void led_pos() {
  for (int i = 0; i < NUM_PR; i++) {
    if (pr_bin[i] == 1) {
      if (i == 0 || i == 4) leds[i + 10] = CRGB(255, 0, 0);
      if (i == 1 || i == 3) leds[i + 10] = CRGB(255, 255, 0);
      if (i == 2) leds[i + 10] = CRGB(0, 0, 255);
    } else {
      leds[i + 10] = CRGB(0, 0, 0); // 消灯
    }
  }
  FastLED.show(); // 表示を更新
}

// フォトリフレクタのセンサ値 pr[]: 0-4095
// フォトリフレクタの正規化値 pr_nrm[]: 0.0-1.0
// フォトリフレクタの2値化 pr_bin[]: 0, 1
void get_pr() {
  for (int i = 0; i < NUM_PR; i++) {
    pr[i] = analogRead(pr_pins[i]);
  }
  for (int i = 0; i < NUM_PR; i++) {
    if (pr[i] < pr_min[i]) pr_min[i] = pr[i];
    if (pr[i] > pr_max[i]) pr_max[i] = pr[i];
    pr_nrm[i] = (pr[i] - pr_min[i]) / (float)(pr_max[i] - pr_min[i]);
    if (pr_nrm[i] > 0.6) pr_bin[i] = 1;
    else pr_bin[i] = 0;
  }
  led_pos();
}

void pr_minmax() {
  for (int i = 0; i < NUM_PR; i++) {
    pr[i] = analogRead(pr_pins[i]);
  }
  for (int i = 0; i < NUM_PR; i++) {
    if (pr[i] < pr_min[i]) pr_min[i] = pr[i];
    if (pr[i] > pr_max[i]) pr_max[i] = pr[i];
  }
}

void pr_print_minmax() {
  Serial.print("int pr_min[PR_NUM] = {");
  for (int i = 0; i < NUM_PR; i++) {
    Serial.print(pr_min[i]);
    if (i < NUM_PR - 1) Serial.print(",");
  }
  Serial.println("};");

  Serial.print("int pr_max[PR_NUM] = {");
  for (int i = 0; i < NUM_PR; i++) {
    Serial.print(pr_max[i]);
    if (i < NUM_PR - 1) Serial.print(",");
  }
  Serial.println("};");
}

void pr_calib() {
  for (int i = 0; i < 80; i++) {
    if ( (i > 10 && i <= 30) || (i > 50 && i <= 70) ) {
      motor(sp, sp, HIGH, LOW);
    } else {
      motor(sp, sp, LOW, HIGH);
    }
    pr_minmax();
    delay(40);
  }
  motor(0, 0, LOW, LOW);
}


//===========================================
// モーター制御の関数
//===========================================
void pd(int pos) {
  static int error_p;
  int error = pos_t - pos; // 目標値と計測位置の差
  int sp_diff = KP * error + KD * (error - error_p); // PD制御：KP*error + KD(error - error_p)
  error_p = error;
  int sp_l = sp - sp_diff; // 左モーターの速度
  int sp_r = sp + sp_diff; // 右モーターの速度

  // 左，右モーターの最小値，最大値を超えないか確認
  if (sp_l < 0)   sp_l = 0;
  if (sp_r < 0)   sp_r = 0;
  if (sp_l > LIMIT) sp_l = LIMIT;
  if (sp_r > LIMIT) sp_r = LIMIT;

  motor(sp_l, sp_r, HIGH, HIGH);
}

//===========================================
// モーターの関数
//===========================================
void motor(int left, int right, int left_c, int right_c) {
  digitalWrite(MOTOR_L_CWCCW, left_c);
  digitalWrite(MOTOR_R_CWCCW, right_c);
  ledcWrite(CH_0, left);
  ledcWrite(CH_1, right);
}

void motor_stop(int time) {
  motor(0, 0, HIGH, HIGH);
  delay(time);
}
